const User = require("../models/user.model");

exports.getStudentsByCollege = async (req, res) => {
  try {
    const { college } = req.params;
    const {
      search = "",
      sortBy = "firstName",
      sortOrder = "asc",
      page = 1,
      limit = 10,
      department,
      status,
      year,
    } = req.query;

    if (!college) {
      return res
        .status(400)
        .json({ success: false, message: "College is required." });
    }

    // Base filter for college and role
    let filter = {
      college: { $regex: new RegExp(`^${college}$`, "i") },
      role: "user",
    };

    // Add department filter if provided
    if (department && department !== "All Departments") {
      filter.department = { $regex: new RegExp(department, "i") };
    }

    // Add status filter if provided
    if (status && status !== "All Status") {
      filter.status = { $regex: new RegExp(status, "i") };
    }

    // Add year filter if provided
    if (year && year !== "All Years") {
      filter.year = year;
    }

    // Handle search functionality
    if (search) {
      const trimmedSearch = search.trim();

      if (trimmedSearch) {
        const searchConditions = [];

        // Name search
        if (trimmedSearch.includes(" ")) {
          const nameParts = trimmedSearch.split(/\s+/);
          searchConditions.push({
            $and: [
              { firstName: { $regex: new RegExp(nameParts[0], "i") } },
              {
                lastName: {
                  $regex: new RegExp(nameParts.slice(1).join(" "), "i"),
                },
              },
            ],
          });
        } else {
          searchConditions.push(
            { firstName: { $regex: new RegExp(trimmedSearch, "i") } },
            { lastName: { $regex: new RegExp(trimmedSearch, "i") } }
          );
        }

        // Other field searches
        searchConditions.push(
          { department: { $regex: new RegExp(trimmedSearch, "i") } },
          { status: { $regex: new RegExp(trimmedSearch, "i") } }
        );

        // Numeric searches
        if (!isNaN(trimmedSearch)) {
          searchConditions.push(
            { year: trimmedSearch },
            { package: parseFloat(trimmedSearch) }
          );
        }

        filter.$or = searchConditions;
      }
    }

    // Sorting options
    const sortOptions = {};
    const order = sortOrder === "asc" ? 1 : -1;

    // Handle different sort fields
    switch (sortBy) {
      case "package":
        sortOptions.package = order;
        break;
      case "year":
        sortOptions.year = order;
        break;
      case "firstName":
        sortOptions.firstName = order;
        break;
      case "lastName":
        sortOptions.lastName = order;
        break;
      case "cgpa":
        sortOptions.cgpa = order;
        break;
      case "status":
        sortOptions.status = order;
        break;
      case "department":
        sortOptions.department = order;
        break;
      default:
        sortOptions.firstName = 1; // Default sort
    }

    const skip = (page - 1) * limit;

    // Execute query with all filters and sorting
    const studentsQuery = User.find(filter)
      .select("-password -savedJobs -appliedJobs")
      .sort(sortOptions)
      .skip(skip)
      .limit(parseInt(limit));

    const students = await studentsQuery.exec();

    // Get total count for pagination
    const totalCount = await User.countDocuments(filter);

    // Stats aggregation
    const statsAggregation = await User.aggregate([
      { $match: filter },
      {
        $project: {
          package: { $toDouble: "$package" },
        },
      },
      {
        $group: {
          _id: null,
          highestPackage: { $max: "$package" },
          averagePackage: { $avg: "$package" },
          count: { $sum: 1 },
        },
      },
    ]);

    const stats = statsAggregation[0] || {
      highestPackage: 0,
      averagePackage: 0,
      count: 0,
    };

    res.status(200).json({
      success: true,
      students,
      stats: {
        highestPackage: stats.highestPackage || 0,
        averagePackage: stats.averagePackage
          ? parseFloat(stats.averagePackage.toFixed(2))
          : 0,
        totalStudents: stats.count || 0,
      },
      pagination: {
        total: totalCount,
        page: parseInt(page),
        limit: parseInt(limit),
        pages: Math.ceil(totalCount / limit),
      },
    });
  } catch (error) {
    console.error("Error fetching students:", error);
    res.status(500).json({ success: false, message: "Server Error" });
  }
};
