const User = require("../models/user.model");
const Job = require("../models/job.model");
const Application = require("../models/application.model");

exports.getJobsByCategory = async (req, res) => {
  try {
    const { category } = req.params;
    const {
      preferedExperience,
      minSalary,
      workModes,
      salary,
      page = 1,
      limit = 6,
    } = req.query;

    // Validate and sanitize pagination parameters
    const sanitizedPage = Math.max(1, parseInt(page));
    const sanitizedLimit = Math.min(50, Math.max(1, parseInt(limit)));

    const regex = new RegExp(`^${category}`, "i");

    // Calculate date 30 days ago
    const thirtyDaysAgo = new Date();
    thirtyDaysAgo.setDate(thirtyDaysAgo.getDate() - 30);

    // Build filter
    let filter = {
      categories: { $elemMatch: { $regex: regex } },
      createdAt: { $gte: thirtyDaysAgo }, // Only show jobs created within the last 30 days
    };

    // Handle preferedExperience filter
    if (preferedExperience) {
      const experienceArray = preferedExperience
        .split(",")
        .map((exp) => exp.trim());
      filter.preferedExperience = { $in: experienceArray };
    }

    // Handle workModes filter
    if (workModes) {
      const workModesArray = workModes.split(",").map((mode) => mode.trim());
      filter.workModes = { $in: workModesArray };
    }

    // Handle minSalary filter in the database query
    // Handle salary filters
    if (minSalary > 0) {
      const userMinSalary = parseInt(minSalary);
      filter.minSalary = { $gte: userMinSalary };
    } else if (salary > 0) {
      const userSalary = parseInt(salary);
      filter.minSalary = { $lte: userSalary };
      filter.maxSalary = { $gte: userSalary };
    }

    // Calculate pagination
    const skip = (sanitizedPage - 1) * sanitizedLimit;

    // Fetch jobs with pagination
    const [jobs, total] = await Promise.all([
      Job.find(filter)
        .populate({
          path: "createdBy",
          select: "companyDescription companyLogo companyName",
        })
        .sort({ createdAt: -1 })
        .skip(skip)
        .limit(sanitizedLimit)
        .lean(),
      Job.countDocuments(filter),
    ]);

    // Calculate total pages
    const totalPages = Math.ceil(total / sanitizedLimit);

    // Ensure current page is valid
    const currentPage = Math.min(sanitizedPage, totalPages || 1);

    // Remove applicants before sending
    const jobsWithoutApplicants = jobs.map((job) => {
      delete job.applicants;
      return job;
    });

    res.status(200).json({
      message: `${category} jobs fetched successfully`,
      jobs: jobsWithoutApplicants,
      pagination: {
        currentPage,
        totalPages,
        totalItems: total,
        itemsPerPage: sanitizedLimit,
        hasNextPage: currentPage < totalPages,
        hasPreviousPage: currentPage > 1,
      },
    });
  } catch (error) {
    console.error("Error fetching jobs:", error);
    res.status(500).json({ message: error.message });
  }
};

exports.getJobs = async (req, res) => {
  try {
    const { title, location } = req.query;

    // Calculate date 30 days ago
    const thirtyDaysAgo = new Date();
    thirtyDaysAgo.setDate(thirtyDaysAgo.getDate() - 30);

    // 1. Fetch all jobs created within the last 30 days
    const jobs = await Job.find({ createdAt: { $gte: thirtyDaysAgo } })
      .populate({
        path: "createdBy",
        select: "companyDescription companyLogo companyName",
      })
      .lean();

    let filteredJobs = jobs;

    // 2. Apply title filter (title OR createdBy.companyName)
    if (title) {
      const titleRegex = new RegExp(title, "i");
      filteredJobs = filteredJobs.filter(
        (job) =>
          titleRegex.test(job.title) ||
          (job.createdBy && titleRegex.test(job.createdBy.companyName))
      );
    }

    // 3. Apply location filter (preferedStates OR preferedCities)
    if (location) {
      const locationRegex = new RegExp(location, "i");
      filteredJobs = filteredJobs.filter(
        (job) =>
          (job.preferedStates &&
            job.preferedStates.some((state) => locationRegex.test(state))) ||
          (job.preferedCities &&
            job.preferedCities.some((city) => locationRegex.test(city)))
      );
    }

    // 4. Remove applicants field
    const jobsWithoutApplicants = filteredJobs.map((job) => {
      delete job.applicants;
      return job;
    });

    res.status(200).json({
      message: "Jobs fetched successfully",
      jobs: jobsWithoutApplicants,
    });
  } catch (error) {
    console.error(error);
    res.status(500).json({ message: error.message });
  }
};

exports.getFeaturedJobs = async (req, res) => {
  try {
    const jobs = await Job.find({ isFeatured: true })
      .populate({
        path: "createdBy",
        select: "companyDescription companyLogo companyName",
      })
      .lean();
    res
      .status(200)
      .json({ message: "Featured jobs fetched successfully", jobs });
  } catch (error) {
    console.error(error);
    res.status(500).json({ message: error.message });
  }
};

exports.createJob = async (req, res) => {
  try {
    if (req.user.role !== "employer") {
      return res.status(403).json({
        code: "unauthorized",
        message: "You are not authorized to create a job",
      });
    }
    const job = await Job.create(req.body);

    res.status(200).json({ message: "Job created successfully", job });
  } catch (error) {
    res.status(500).json({
      message: "Something went wrong when creating a job: ",
      error: error.message,
    });
  }
};

exports.updateJob = async (req, res) => {
  try {
    if (req.user.role !== "employer" || req.user.id !== req.body.createdBy) {
      return res.status(403).json({
        code: "unauthorized",
        message: "You are not authorized to update a job",
      });
    }
    const { id } = req.params;
    const { body } = req;
    const job = await Job.findByIdAndUpdate(id, body, { new: true });
    res.status(200).json({ message: "Job updated successfully", job });
  } catch (error) {
    res.status(500).json({
      message: "Something went wrong when updating a job: ",
      error: error.message,
    });
  }
};

exports.deleteJob = async (req, res) => {
  try {
    const { id } = req.params;
    const job = await Job.findByIdAndDelete(id);
    res.status(200).json({ message: "Job deleted successfully", job });
  } catch (error) {
    res.status(500).json({
      message: "Something went wrong when deleting a job: ",
      error: error.message,
    });
  }
};

exports.applyJob = async (req, res) => {
  try {
    const { id } = req.params; // job id
    const { resume } = req.body; // resume link must be sent from frontend

    if (!resume) {
      return res.status(400).json({ message: "Resume is required" });
    }
    // Check if already applied
    const existingApplication = await Application.findOne({
      user: req.user.id,
      job: id,
    });

    if (existingApplication) {
      return res
        .status(400)
        .json({ message: "You have already applied to this job" });
    }

    // Create new application
    const application = new Application({
      user: req.user.id,
      job: id,
      resume: resume,
    });

    await application.save();

    await Job.findByIdAndUpdate(id, {
      //   $push: { applicants: { userId: req.user.id, appliedAt: new Date() } },
      $inc: { applicantsCount: 1 },
    });

    res.status(201).json({ message: "Job applied successfully", application });
  } catch (error) {
    res.status(500).json({
      message: "Something went wrong when applying to a job",
      error: error.message,
    });
  }
};

exports.toggleSaveJob = async (req, res) => {
  try {
    const { id } = req.params;
    const user = await User.findById(req.user.id);

    // Check if job is already saved
    const jobIndex = user.savedJobs.findIndex(
      (job) => job.jobId.toString() === id
    );

    if (jobIndex > -1) {
      // Job already saved, so remove it
      await User.findByIdAndUpdate(req.user.id, {
        $pull: { savedJobs: { jobId: id } },
      });
      res.status(200).json({ message: "Job unsaved successfully" });
    } else {
      // Job not saved, so save it
      await User.findByIdAndUpdate(req.user.id, {
        $push: { savedJobs: { jobId: id, savedAt: new Date() } },
      });
      res.status(200).json({ message: "Job saved successfully" });
    }
  } catch (error) {
    res.status(500).json({
      message: "Something went wrong when saving/unsaving job: ",
      error: error.message,
    });
  }
};

exports.getJobById = async (req, res) => {
  try {
    const { id } = req.params;
    const job = await Job.findById(id).populate({
      path: "createdBy",
      select: "companyDescription companyLogo companyName",
    });
    res.status(200).json({ message: "Job fetched successfully", job });
  } catch (error) {
    res.status(500).json({
      message: "Something went wrong when fetching a job: ",
      error: error.message,
    });
  }
};
